<?php

namespace App\Utils;

use App\AccountTransaction;
use App\BusinessLocation;
use App\Contact;
use App\Events\TransactionPaymentAdded;
use App\Payment;
use App\TransactionPayment;
use App\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;
use Yajra\DataTables\Facades\DataTables;

class HandcashUtil extends Util
{
    /**
     * Process transaction payment with hand cash tracking
     * 
     * @param int|null $contact_id Contact ID
     * @param int|null $transaction_id Transaction ID
     * @param string $type Payment type (purchase_payment, sell_payment, etc.)
     * @param string $payment_type Payment method (credit, debit)
     * @param float $amount Payment amount
     * @param string|null $note Payment note
     * @param int|null $update Payment ID to update (null for new payment)
     * @param int|null $location_id Location ID
     * @param bool $allowed0Amount Allow zero amount payments
     * @return bool Success status
     */
    public function transactionPayment(
        $contact_id = null, 
        $transaction_id = null, 
        $type, 
        $payment_type, 
        $amount, 
        $note = null, 
        $update = null, 
        $location_id = null, 
        $allowed0Amount = false
    ) {
        try {
            DB::beginTransaction();

            // Normalize payment types
            $type = $this->normalizePaymentType($type, $payment_type);
            
            // Get previous due amount
            $previews_due = request()->input('previews_due', 0);
            
            // Prepare payment data
            $data = $this->preparePaymentData($contact_id, $transaction_id, $type, $payment_type, $amount, $previews_due, $note, $location_id);
            
            if ($update !== null) {
                $result = $this->updatePayment($update, $data, $amount, $previews_due);
            } else {
                $result = $this->createPayment($data, $amount, $payment_type, $allowed0Amount);
            }
            
            DB::commit();
            return $result;
            
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Transaction payment error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Delete transaction payment and update hand cash
     * 
     * @param int|null $contact_id Contact ID
     * @param int|null $transaction_id Transaction ID
     * @return bool Success status
     */
    public function transactionPaymentDelete($contact_id = null, $transaction_id = null)
    {
        try {
            DB::beginTransaction();

            $payment = Payment::query()
                ->where('contact_id', $contact_id)
                ->where('transaction_id', $transaction_id)
                ->latest('id')
                ->first();
            
            if (!$payment) {
                DB::rollBack();
                return false;
            }

            // Update subsequent payments' hand_cash
            $this->updateSubsequentPaymentsAfterDeletion($payment);
            
            // Delete the payment
            $payment->delete();
            
            DB::commit();
            return true;
            
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Transaction payment deletion error: ' . $e->getMessage());
            return false;
        }
    }

    public function payPreviousDue($reqDenominations, $contact_id, $previews_due = 0)
    {
        $pos_settings = ! empty(session()->get('business.pos_settings')) ? json_decode(session()->get('business.pos_settings'), true) : [];
        $business_id = session()->get('user.business_id');

        $tp = $this->payDueContact($contact_id, $previews_due);
        $pos_settings = ! empty(session()->get('business.pos_settings')) ? json_decode(session()->get('business.pos_settings'), true) : [];
        $enable_cash_denomination_for_payment_methods = ! empty($pos_settings['enable_cash_denomination_for_payment_methods']) ? $pos_settings['enable_cash_denomination_for_payment_methods'] : [];
        
        //add cash denomination
        if (in_array($tp->method, $enable_cash_denomination_for_payment_methods) && ! empty($reqDenominations) && ! empty($pos_settings['enable_cash_denomination_on']) && $pos_settings['enable_cash_denomination_on'] == 'all_screens') {
            $denominations = [];
            foreach ($reqDenominations as $key => $value) {
                if (! empty($value)) {
                    $denominations[] = [
                        'business_id' => $business_id,
                        'amount' => $key,
                        'total_count' => $value,
                    ];
                }
            }
            if (! empty($denominations)) {
                $tp->denominations()->createMany($denominations);
            }
        }
    }

    /**
     * Get current hand cash balance for staff
     * 
     * @param int|null $business_id Business ID
     * @param int|null $location_id Location ID
     * @param int|null $staff_id Staff ID
     * @return float Current hand cash balance
     */
    public function getCurrentHandCash($business_id = null, $location_id = null, $staff_id = null)
    {
        $business_id = $business_id ?? request()->session()->get('business.id');
        $location_id = $location_id ?? 1;
        $staff_id = $staff_id ?? auth()->id();

        return DB::table('payments')
            ->where('business_id', $business_id)
            ->where('location_id', $location_id)
            ->where('staff_name', $staff_id)
            ->latest('id')
            ->value('hand_cash') ?? 0;
    }

    /**
     * Get hand cash history for a date range
     * 
     * @param string $start_date Start date
     * @param string $end_date End date
     * @param int|null $staff_id Staff ID
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getHandCashHistory($start_date, $end_date, $staff_id = null)
    {
        $staff_id = $staff_id ?? auth()->id();
        $business_id = request()->session()->get('business.id');

        return Payment::query()
            ->where('business_id', $business_id)
            ->where('staff_name', $staff_id)
            ->whereBetween('date', [$start_date, $end_date])
            ->orderBy('date', 'desc')
            ->orderBy('id', 'desc')
            ->get(['id', 'date', 'type', 'payment_type', 'amount', 'hand_cash', 'note']);
    }

    /**
     * Normalize payment type based on payment method
     * 
     * @param string $type Original type
     * @param string $payment_type Payment method
     * @return string Normalized type
     */
    private function normalizePaymentType($type, $payment_type)
    {
        if ($type == 'purchase_payment' && $payment_type == 'credit') {
            return 'purchase_return';
        }
        
        if ($type == 'sell_payment' && $payment_type == 'debit') {
            return 'sell_return';
        }
        
        return $type;
    }

    /**
     * Prepare payment data array
     * 
     * @param mixed $contact_id Contact ID
     * @param mixed $transaction_id Transaction ID
     * @param string $type Payment type
     * @param string $payment_type Payment method
     * @param float $amount Amount
     * @param float $previews_due Previous due amount
     * @param string|null $note Note
     * @param int|null $location_id Location ID
     * @return array Payment data
     */
    private function preparePaymentData($contact_id, $transaction_id, $type, $payment_type, $amount, $previews_due, $note, $location_id)
    {
        $data = [
            'business_id' => request()->session()->get('business.id'),
            'location_id' => $location_id ?? 1,
            'contact_id' => $contact_id,
            'transaction_id' => $transaction_id,
            'staff_name' => auth()->id(),
            'type' => $type,
            'payment_type' => $payment_type,
            'amount' => $amount,
            'previews_due' => $previews_due,
            'note' => $note,
        ];

        // Set transaction date
        if (!empty(request()->input('transaction_date'))) {
            $data['date'] = $this->uf_date(request()->input('transaction_date'), true);
        } else {
            $data['date'] = now();
        }

        return $data;
    }

    /**
     * Update existing payment
     * 
     * @param int $payment_id Payment ID to update
     * @param array $data Payment data
     * @param float $amount New amount
     * @param float $previews_due Previous due amount
     * @return bool Success status
     */
    private function updatePayment($payment_id, $data, $amount, $previews_due)
    {
        $payment = Payment::query()->find($payment_id);
        
        if (!$payment) {
            return false;
        }

        // Calculate hand cash based on previous payment
        $data['hand_cash'] = $this->calculateHandCashForUpdate($payment, $data, $amount, $previews_due);
        
        // Update the payment
        $payment->update($data);

        // Update all subsequent payments
        $this->updateSubsequentPayments($payment);

        return true;
    }

    /**
     * Create new payment
     * 
     * @param array $data Payment data
     * @param float $amount Payment amount
     * @param string $payment_type Payment type
     * @param bool $allowed0Amount Allow zero amount
     * @return bool Success status
     */
    private function createPayment($data, $amount, $payment_type, $allowed0Amount)
    {
        if ($amount <= 0 && !$allowed0Amount) {
            return false;
        }

        // Get current hand cash balance
        $current_hand_cash = $this->getCurrentHandCash($data['business_id'], $data['location_id'], $data['staff_name']);
        
        // Calculate new hand cash balance
        if ($payment_type == 'credit') {
            $data['hand_cash'] = $current_hand_cash + $amount;
        } else {
            $data['hand_cash'] = $current_hand_cash - $amount;
        }
        
        Payment::query()->create($data);
        return true;
    }

    /**
     * Calculate hand cash for payment update
     * 
     * @param Payment $payment Payment being updated
     * @param array $data New payment data
     * @param float $amount New amount
     * @param float $previews_due Previous due amount
     * @return float New hand cash balance
     */
    private function calculateHandCashForUpdate($payment, $data, $amount, $previews_due)
    {
        // Get previous payment's hand cash
        $previous_hand_cash = DB::table('payments')
            ->where('id', '<', $payment->id)
            ->where('business_id', $data['business_id'])
            ->where('location_id', $data['location_id'])
            ->where('staff_name', auth()->id())
            ->latest('id')
            ->value('hand_cash') ?? 0;

        // Calculate new hand cash based on payment type
        if ($payment->payment_type == 'credit') {
            return $previous_hand_cash + $amount + $previews_due;
        } else {
            return $previous_hand_cash - $amount;
        }
    }

    /**
     * Update all payments after a specific payment
     * 
     * @param Payment $payment Reference payment
     * @return void
     */
    private function updateSubsequentPayments($payment)
    {
        $nextPayments = Payment::query()
            ->where('id', '>', $payment->id)
            ->where('business_id', $payment->business_id)
            ->where('location_id', $payment->location_id)
            ->where('staff_name', $payment->staff_name)
            ->orderBy('id')
            ->get();

        $previousPayment = $payment;

        foreach ($nextPayments as $nextPayment) {
            $hand_cash = $previousPayment->hand_cash;
            
            if ($nextPayment->payment_type == 'credit') {
                $hand_cash += $nextPayment->amount;
            } else {
                $hand_cash -= $nextPayment->amount;
            }
            
            $nextPayment->update(['hand_cash' => $hand_cash]);
            $previousPayment = $nextPayment;
        }
    }

    /**
     * Update subsequent payments after deletion
     * 
     * @param Payment $deletedPayment Payment being deleted
     * @return void
     */
    private function updateSubsequentPaymentsAfterDeletion($deletedPayment)
    {
        $subsequentPayments = Payment::query()
            ->where('id', '>', $deletedPayment->id)
            ->where('business_id', $deletedPayment->business_id)
            ->where('location_id', $deletedPayment->location_id)
            ->where('staff_name', $deletedPayment->staff_name)
            ->orderBy('id')
            ->get();

        // Get the hand cash amount before the deleted payment
        $baseHandCash = DB::table('payments')
            ->where('id', '<', $deletedPayment->id)
            ->where('business_id', $deletedPayment->business_id)
            ->where('location_id', $deletedPayment->location_id)
            ->where('staff_name', $deletedPayment->staff_name)
            ->latest('id')
            ->value('hand_cash') ?? 0;

        $runningHandCash = $baseHandCash;

        foreach ($subsequentPayments as $payment) {
            if ($payment->payment_type == 'credit') {
                $runningHandCash += $payment->amount;
            } else {
                $runningHandCash -= $payment->amount;
            }
            
            $payment->update(['hand_cash' => $runningHandCash]);
        }
    }

    /**
     * Validate hand cash integrity for a staff member
     * 
     * @param int|null $staff_id Staff ID
     * @param int|null $business_id Business ID
     * @param int|null $location_id Location ID
     * @return array Validation result
     */
    private function validateHandCashIntegrity($staff_id = null, $business_id = null, $location_id = null)
    {
        $staff_id = $staff_id ?? auth()->id();
        $business_id = $business_id ?? request()->session()->get('business.id');
        $location_id = $location_id ?? 1;

        $payments = Payment::query()
            ->where('staff_name', $staff_id)
            ->where('business_id', $business_id)
            ->where('location_id', $location_id)
            ->orderBy('id')
            ->get();

        $errors = [];
        $expected_hand_cash = 0;

        foreach ($payments as $payment) {
            if ($payment->payment_type == 'credit') {
                $expected_hand_cash += $payment->amount;
            } else {
                $expected_hand_cash -= $payment->amount;
            }

            if ($payment->hand_cash != $expected_hand_cash) {
                $errors[] = [
                    'payment_id' => $payment->id,
                    'expected' => $expected_hand_cash,
                    'actual' => $payment->hand_cash,
                    'difference' => $payment->hand_cash - $expected_hand_cash
                ];
            }
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors,
            'total_payments' => $payments->count(),
            'final_balance' => $expected_hand_cash
        ];
    }

    /**
     * Process payments and update hand cash
     */
    public function processPaymentsWithHandCash($request, $transaction, $update = null)
    {
        $bkashAmount = 0;
        $nagadAmount = 0;
        $cashAmount = 0;

        // Calculate payment amounts by method
        foreach ($request->payment as $payment) {
            $amount = $this->num_uf($payment['amount'] ?? 0);
            
            switch ($payment['method']) {
                case 'cash':
                    $cashAmount += $amount;
                    break;
                case 'custom_pay_1':
                    $bkashAmount += $amount;
                    break;
                case 'custom_pay_2':
                    $nagadAmount += $amount;
                    break;
            }
        }
        
        $totalCashAmount = $cashAmount + $this->num_uf($request->previews_due ?? 0) - ($request->change_return ?? 0);

        if ($cashAmount > 0) {
            $this->transactionPayment(
                contact_id: $transaction->contact_id, 
                transaction_id: $transaction->id, 
                type: 'sell', 
                payment_type: 'credit', 
                amount: $totalCashAmount, 
                note: $request->sale_note,
                update: $update,
                location_id: $request->location_id,
            );
        }
    }

    /**
     * Cash in hand
     */
    public function inHandCash($start = null, $end = null, $location_id = null)
    {
        $enableMultipleCash = $this->isModuleEnabled('multiple_cash');
        $locations = BusinessLocation::query()->pluck('id');

        $handCash = 0;
        if($enableMultipleCash) {
            if (isMainAdmin()) {
                foreach (User::query()->pluck('id')->toArray() as $id) {
                    $handCash += DB::table('payments')->when($location_id != null, fn($q) => $q->where('location_id', $location_id))->where('staff_name', $id)->latest('id')->value('hand_cash') ?? 0;
                }
            }
            else {
                $handCash = DB::table('payments')->when($location_id != null, fn($q) => $q->where('location_id', $location_id))->where('staff_name', auth()->id())->latest('id')->value('hand_cash') ?? 0;
            }
        }
        else {
            if(empty($location_id)) {
                foreach ($locations as $location) {
                    $handCash += DB::table('payments')->where('location_id', $location)->latest('id')->value('hand_cash') ?? 0; 
                }
            }
            else {
                $handCash = DB::table('payments')->where('location_id', $location_id)->latest('id')->value('hand_cash') ?? 0;
            }
        }
        
        return $handCash;
    }

    public function getPayments()
    {
        if (request()->ajax()) {
            $currentHandCashGet = false;
            $currentHandCash = $this->num_f(0, true);
            $query = Payment::query()->with('contact:id,type,supplier_business_name,name', 'user:id,surname,first_name,last_name', 'location:id,name');
            return DataTables::eloquent($query)
                ->addIndexColumn()
                ->filter(function($q) {
                    if(!empty(request()->get('location_id'))) {
                        $q->where('location_id', request()->get('location_id'));
                    }
                    if(!empty(request()->get('contact_id'))) {
                        $q->where('contact_id', request()->get('contact_id'));
                    }
                    if(!isMainAdmin()) {
                        $q->where('staff_name', auth()->id());
                    }
                    else {
                        if(!empty(request()->get('staff_name'))) {
                            $q->where('staff_name', request()->get('staff_name'));
                        }
                    }
        
                    if (! empty(request()->start_date) && ! empty(request()->end_date)) {
                        $start = request()->start_date;
                        $end = request()->end_date;
                        $q->whereDate('date', '>=', $start)->whereDate('date', '<=', $end);
                    }
                }, true)
                ->addColumn('contact_name', function($data) {
                    $name = $data->contact?->name;
                    $supplier_business_name = $data->contact?->supplier_business_name;
                    if($supplier_business_name != null) {
                        $name .= "-$supplier_business_name";
                    }
                    return $name;
                })
                ->addColumn('current_hand_cash', function($data) use(&$currentHandCashGet, &$currentHandCash) {
                    if(!$currentHandCashGet) {
                        $currentHandCashGet = true;
                        $currentHandCash = $this->num_f($this->inHandCash(request()->start_date, request()->end_date, request()->get('location_id')), true);
                    }
                    return $currentHandCash;
                })
                ->addColumn('user_name', fn($data) => "{$data->user?->surname} {$data->user?->first_name} {$data->user?->last_name}")
                ->editColumn('type', fn($q) => ucwords(str_replace('_', ' ', $q->type)))
                ->editColumn('date', fn($q) => $this->format_date($q->date, true))
                // ->editColumn('amount', fn($q) => $this->num_f($q->amount, true))
                ->editColumn(
                    'amount',
                    '<span class="final-total" data-orig-value="{{$amount}}">@format_currency($amount)</span>'
                )
                ->editColumn(
                    'hand_cash',
                    '<span class="final-total" data-orig-value="{{$hand_cash}}">@format_currency($hand_cash)</span>'
                )
                ->addColumn(
                    'action',
                    '@if(auth()->user()->can("edit_purchase_payment") && auth()->user()->can("edit_sell_payment"))
                        <button data-href="{{action(\'App\Http\Controllers\TransactionPaymentController@edit\', [$id])}}?payment=1" class="btn btn-xs btn-primary edit_payment_button"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</button>
                    @endif'
                )
                ->rawColumns(['amount', 'hand_cash', 'action'])
                ->make(true);
                // &nbsp;
                // @if(auth()->user()->can("delete_purchase_payment") && auth()->user()->can("delete_sell_payment"))
                //     <button data-href="{{action(\'App\Http\Controllers\TransactionPaymentController@destroy\', [$id])}}?payment=1" class="btn btn-xs btn-danger delete_payment_button"><i class="glyphicon glyphicon-trash"></i> @lang("messages.delete")</button>
                // @endif
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::contactDropdown($business_id);
        $users = User::forDropdown($business_id);
        $business_locations = BusinessLocation::forDropdown($business_id);

        return view('transaction_payment.index', compact('customers', 'users', 'business_locations'));
    }

    public function payDueContact($contact_id, $amount, $format_data = true)
    {
        $business_id = auth()->user()->business_id;
        $inputs = [
            'amount' => $amount,
            'method' => 'cash',
            'account_id' => null,
            'is_reverse' => false,
            'card_type' => 'credit',
            'paid_on'   => now()->format('Y-m-d H:i:s'),
        ];

        //payment type option is available on pay contact modal
        $is_reverse = $inputs['is_reverse'] == 1 ? true : false;

        $payment_types = $this->payment_types();

        if (! array_key_exists($inputs['method'], $payment_types)) {
            throw new Exception('Payment method not found');
        }
        if ($format_data) {
            $inputs['amount'] = $this->num_uf($inputs['amount']);
        }

        $inputs['created_by'] = auth()->user()->id;
        $inputs['payment_for'] = $contact_id;
        $inputs['business_id'] = $business_id;

        if (! $is_reverse) {
            $inputs['is_advance'] = 1;
        }

        $contact = Contact::query()->where('business_id', $business_id)->findOrFail($contact_id);

        $due_payment_type = 'sell';

        $prefix_type = '';
        if ($contact->type == 'customer') {
            $prefix_type = 'sell_payment';
        } elseif ($contact->type == 'supplier') {
            $prefix_type = 'purchase_payment';
        }
        $ref_count = $this->setAndGetReferenceCount($prefix_type, $business_id);
        //Generate reference number
        $payment_ref_no = $this->generateReferenceNumber($prefix_type, $ref_count, $business_id);

        $inputs['payment_ref_no'] = $payment_ref_no;


        //get payment type (creditor debit)
        $payment_type = AccountTransaction::getAccountTransactionType($due_payment_type);

        //if reverse payment
        if ($due_payment_type == 'sell' && $is_reverse) {
            $payment_type = 'debit';
        }
        if ($due_payment_type == 'purchase' && $is_reverse) {
            $payment_type = 'credit';
        }

        $inputs['payment_type'] = $payment_type;

        
        $parent_payment = TransactionPayment::create($inputs);

        $inputs['transaction_type'] = $due_payment_type;

        event(new TransactionPaymentAdded($parent_payment, $inputs));

        $transactionUtil = new TransactionUtil;
        //Distribute above payment among unpaid transactions
        if (! $is_reverse) {
            $excess_amount = $transactionUtil->payAtOnce($parent_payment, $due_payment_type);
        }
        //Update excess amount
        if (! empty($excess_amount)) {
            $transactionUtil->updateContactBalance($contact, $excess_amount);
        }

        return $parent_payment;
    }
}